function CDX = import_index_and_tranche_data_citi(file_name)
% --------------------------------------------------------------------------------------------------
% Imports historical index and tranche prices data from Citi.
% --------------------------------------------------------------------------------------------------
% sample call: CDX_NA_IG_citi = import_index_and_tranche_data_citi('c:/thesis/data/citi/Tranche-data.xls');
% --------------------------------------------------------------------------------------------------

% Import data
[num, txt] = xlsread(file_name, 2, 'A5:AH1000');
dates = datenum(txt(1:end,1));

% Define empty CDX structure
CDX = struct( ...
    'dates', {{[] [] []}}, 'T', {{[] [] []}}, 'cut_offs', {[0.03 0.07 0.1 0.15 0.3]}, ...
    'index_market_price', {{[] [] []}}, ...
    'tranche_market_price_mid', {{[] [] []}}, ...
    'tranche_market_upfront_mid', {{[] [] []}}, ...
    'delta_tr_idx_GC', {{[] [] []}}, ...
    'portfolio', [], ...
    'index_members', {repmat(NaN, length(10), 0)} ...
);

%%%%%%%
% 5YR %
%%%%%%%

% Allocate memory
index_market_price = num(:,1);
market_upfront_mid = zeros(length(dates), 5);
market_price_mid = zeros(length(dates), 5);
market_price_mid(:,1) = 500;

% Impute data
market_upfront_mid(:,1) = num(:,2) / 100;
market_price_mid(:,2:5) = num(:,4:2:10);

% Clean hedges
delta = num(:,3:2:11);
delta(isnan(delta)) = 0;
delta(delta > 99) = 0;

% For each data, determine time to maturity of index
maturities = match_index_maturity_dates(dates, 4.75);
T = zeros(length(dates), 1);
for i=1:length(T)
    T(i) = date_diff_years(dates(i), maturities(i));
end

% Copy into CDX structure
CDX.dates{1} = dates;
CDX.index_market_price{1} = index_market_price;
CDX.tranche_market_price_mid{1} = market_price_mid;
CDX.tranche_market_upfront_mid{1} = market_upfront_mid;
CDX.delta_tr_idx_GC{1} = delta;
CDX.T{1} = T;

%%%%%%%
% 7YR %
%%%%%%%

% Allocate memory
index_market_price = num(:,23);
used_range = find(~isnan(index_market_price));
index_market_price = index_market_price(used_range);
used_dates = dates(used_range);
market_upfront_mid = zeros(length(used_dates), 5);
market_price_mid = zeros(length(used_dates), 5);
market_price_mid(:,1) = 500;

% Impute data
market_upfront_mid(:,1) = num(used_range,24);
market_price_mid(:,2:5) = num(used_range,26:2:32);

% Clean hedges
delta = num(used_range,25:2:33);
delta(isnan(delta)) = 0;
delta(delta > 99) = 0;

% For each data, determine time to maturity of index
maturities = match_index_maturity_dates(used_dates, 6.75);
T = zeros(length(used_dates), 1);
for i=1:length(T)
    T(i) = date_diff_years(used_dates(i), maturities(i));
end

% Copy into CDX structure
CDX.dates{2} = used_dates;
CDX.index_market_price{2} = index_market_price;
CDX.tranche_market_price_mid{2} = market_price_mid;
CDX.tranche_market_upfront_mid{2} = market_upfront_mid;
CDX.delta_tr_idx_GC{2} = delta;
CDX.T{2} = T;

%%%%%%%%
% 10YR %
%%%%%%%%

% Allocate memory
index_market_price = num(:,12);
used_range = find(~isnan(index_market_price));
index_market_price = index_market_price(used_range);
used_dates = dates(used_range);
market_upfront_mid = zeros(length(used_dates), 5);
market_price_mid = zeros(length(used_dates), 5);
market_price_mid(:,1) = 500;

% Impute data
market_upfront_mid(:,1) = num(used_range,13);
market_price_mid(:,2:5) = num(used_range,15:2:21);

% Clean hedges
delta = num(used_range,14:2:22);
delta(isnan(delta)) = 0;
delta(delta > 99) = 0;

% For each data, determine time to maturity of index
maturities = match_index_maturity_dates(used_dates, 9.75);
T = zeros(length(used_dates), 1);
for i=1:length(T)
    T(i) = date_diff_years(used_dates(i), maturities(i));
end

% Copy into CDX structure
CDX.dates{3} = used_dates;
CDX.index_market_price{3} = index_market_price;
CDX.tranche_market_price_mid{3} = market_price_mid;
CDX.tranche_market_upfront_mid{3} = market_upfront_mid;
CDX.delta_tr_idx_GC{3} = delta;
CDX.T{3} = T;